let quizData = JSON.parse(localStorage.getItem('quizData')) || []
let targetTiebreaker = parseInt(localStorage.getItem('targetTiebreaker')) || 0
let showTiebreakerTarget = JSON.parse(localStorage.getItem('showTiebreakerTarget')) || false

document.addEventListener('DOMContentLoaded', function () {
    const editButton = document.querySelector('#editButton')
    const navbarControls = document.querySelector('#navbar-controls')
    let targetTiebreaker = parseInt(localStorage.getItem('targetTiebreaker')) || 0
    function calculateTotal(scores, tiebreaker) {
        return scores.reduce((sum, score) => sum + score, 0)
    }

    function saveToLocalStorage() {
        localStorage.setItem('quizData', JSON.stringify(quizData))
    }

    function clearData() {
        quizData = []
        saveToLocalStorage()
        renderTable()
    }

    function updateTargetTiebreaker(value) {
        targetTiebreaker = parseInt(value)
        localStorage.setItem('targetTiebreaker', targetTiebreaker)
        renderTable()
    }

    function findClosestTiebreaker() {
        if (quizData.length === 0) return -1
        return quizData.reduce((closest, team, index, arr) => {
            if (closest === -1) return index
            const currentDiff = Math.abs(team.tiebreaker - targetTiebreaker)
            const closestDiff = Math.abs(arr[closest].tiebreaker - targetTiebreaker)
            return currentDiff < closestDiff ? index : closest
        }, -1)
    }
    function renderTable() {
        const tableBody = document.getElementById('quiz-table-body')
        tableBody.innerHTML = ''
        
        const isEditing = navbarControls.style.display !== 'none'
        renderTiebreakerHeader()
        
        quizData.sort((a, b) => b.total - a.total)
        const closestIndex = showTiebreakerTarget ? findClosestTiebreaker() : -1
        
        quizData.forEach((team, index) => {
            const row = document.createElement('tr')
            const isClosest = index === closestIndex
            
            // Medal colors only for teams with points
            if (team.total > 0) {
                if (index === 0) row.classList.add('medal-gold')
                if (index === 1) row.classList.add('medal-silver')
                if (index === 2) row.classList.add('medal-bronze')
            }
            
            // Tiebreaker highlighting only for 4th place and below
            if (isClosest && showTiebreakerTarget) {
                row.classList.add('table-success')
                row.style.border = '2px solid #198754'
            }

            if (isEditing) {
                row.innerHTML = `
                    <td>
                        <div class="d-flex align-items-center">
                            <input type="text" class="form-control me-2" value="${team.team}" 
                                onchange="updateTeamName(${index}, this.value)">
                            <button class="btn btn-danger" onclick="deleteTeam(${index})">Usuń</button>
                        </div>
                    </td>
                    ${team.scores.map((score, scoreIndex) => `
                        <td>
                            <input type="number" class="form-control" value="${score}" min="0" max="5"
                                onchange="updateScore(${index}, ${scoreIndex}, this.value)">
                        </td>
                    `).join('')}
                    <td>
                        <input type="number" class="form-control" value="${team.tiebreaker}" min="0" max="10"
                            onchange="updateTiebreaker(${index}, this.value)">
                    </td>
                    <td>${team.total}</td>
                `            } else {
                row.innerHTML = `
                      <td>${team.team}</td>
                      ${team.scores.map(score => `<td>${score}</td>`).join('')}
                      <td>${team.tiebreaker}</td>
                      <td>${team.total}</td>
                  `
            }
            tableBody.appendChild(row)
        })
    }

    function findClosestTiebreaker() {
        if (!showTiebreakerTarget || quizData.length <= 3) return -1

        const allTeamsHaveTiebreaker = quizData.every(team => team.tiebreaker !== 0)
        if (!allTeamsHaveTiebreaker) return -1
        
        let bestDiff = Infinity
        let bestIndex = -1
        
        for (let i = 3; i < quizData.length; i++) {
            const diff = Math.abs(quizData[i].tiebreaker - targetTiebreaker)
            if (diff < bestDiff) {
                bestDiff = diff
                bestIndex = i
            }
        }
        
        return bestIndex
    }

    function toggleEditMode() {
        const isEditing = navbarControls.style.display !== 'none'
        navbarControls.style.display = isEditing ? 'none' : 'block'

        if (!isEditing) {
            navbarControls.innerHTML = `
    <div class="d-flex align-items-center">
        <button id="addTeamButton" class="btn btn-success fs-4 me-2">Dodaj drużynę</button>
        <button id="clearDataButton" class="btn btn-warning fs-4 me-2">Wyczyść dane</button>
        <button id="exportButton" class="btn btn-info fs-4 me-2" onclick="exportToCSV()">Eksport CSV</button>
        <div class="input-group" style="width: auto;">
            <div class="input-group-text">
                <input class="form-check-input mt-0" type="checkbox" ${showTiebreakerTarget ? 'checked' : ''} onchange="toggleTiebreakerTarget(this.checked)">
            </div>
            <input type="number" class="form-control" value="${targetTiebreaker}" onchange="updateTargetTiebreaker(this.value)">
        </div>
    </div>
`;
            document.querySelector('#addTeamButton').addEventListener('click', addTeam)
            document.querySelector('#clearDataButton').addEventListener('click', clearData)
        }

        renderTable()
    } function addTeam() {
        const newTeam = {
            team: `Team ${quizData.length + 1}`,
            scores: Array(10).fill(0),
            tiebreaker: 0,
            total: 0
        }
        quizData.push(newTeam)
        saveToLocalStorage()
        renderTable()
    }

    window.updateTeamName = function (index, newName) {
        quizData[index].team = newName
        saveToLocalStorage()
        renderTable()
    }

    window.deleteTeam = function (index) {
        quizData.splice(index, 1)
        saveToLocalStorage()
        renderTable()
    }

    window.updateScore = function (teamIndex, scoreIndex, value) {
        quizData[teamIndex].scores[scoreIndex] = Number(value)
        quizData[teamIndex].total = calculateTotal(quizData[teamIndex].scores, quizData[teamIndex].tiebreaker)
        saveToLocalStorage()
        renderTable()
    }

    window.updateTiebreaker = function (teamIndex, value) {
        quizData[teamIndex].tiebreaker = Number(value)
        quizData[teamIndex].total = calculateTotal(quizData[teamIndex].scores, quizData[teamIndex].tiebreaker)
        saveToLocalStorage()
        renderTable()
    }

    window.updateTargetTiebreaker = updateTargetTiebreaker
    window.toggleTiebreakerTarget = toggleTiebreakerTarget
    window.renderTable = renderTable

    editButton.addEventListener('click', toggleEditMode)
    document.querySelector('#addTeamButton').addEventListener('click', addTeam)
    document.querySelector('#clearDataButton').addEventListener('click', clearData)

    navbarControls.style.display = 'none'
    renderTable()
})

function updateTargetTiebreaker(value) {
    targetTiebreaker = parseInt(value)
    localStorage.setItem('targetTiebreaker', targetTiebreaker)
    renderTable()
}

function toggleTiebreakerTarget(checked) {
    showTiebreakerTarget = checked
    localStorage.setItem('showTiebreakerTarget', checked)
    renderTable()
}
function findClosestTiebreaker() {
    if (!showTiebreakerTarget || quizData.length <= 3) return -1

    // Sort by total points first
    quizData.sort((a, b) => b.total - a.total)

    // Find closest tiebreaker only among teams in positions 4+
    let bestDiff = Math.abs(quizData[3].tiebreaker - targetTiebreaker)
    let bestIndex = 3

    for (let i = 3; i < quizData.length; i++) {
        const currentDiff = Math.abs(quizData[i].tiebreaker - targetTiebreaker)
        if (currentDiff <= bestDiff) {
            bestDiff = currentDiff
            bestIndex = i
        }
    }

    return bestIndex
}

function renderTiebreakerHeader() {
    const header = document.getElementById('tiebreaker-header')
    header.innerHTML = showTiebreakerTarget ? `Dogrywka (${targetTiebreaker})` : 'Dogrywka'
}

function exportToCSV() {
    const today = new Date().toISOString().split('T')[0]
    const headers = ['Nazwa Drużyny', ...Array(10).fill(0).map((_, i) => `K${i+1}`), 'Dogrywka', 'Suma']
    const csvContent = [
        headers.join(','),
        ...quizData.map(team => [
            team.team,
            ...team.scores,
            team.tiebreaker,
            team.total
        ].join(','))
    ].join('\n')
    
    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' })
    const link = document.createElement('a')
    link.href = URL.createObjectURL(blob)
    link.download = `quiz_${today}.csv`
    link.click()
}